<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class WC_Store_Catalog_PDF_Download_Admin {
	private static $_this;

	private $_settings_tab_id = 'scpdfd';

	/**
	 * Init
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		self::$_this = $this;

		add_filter( 'woocommerce_get_sections_products', array( $this, 'add_settings_section' ) );

		add_filter( 'woocommerce_get_settings_products', array( $this, 'add_settings' ), 10, 2 );

		add_action( 'woocommerce_settings_save_products', array( $this, 'save_settings' ) );

		add_action( 'woocommerce_admin_field_woocommerce_store_catalog_pdf_download_settings', array( $this, 'get_settings' ) );

		add_action( 'admin_enqueue_scripts', array( $this, 'load_scripts' ) );
	}

	/**
	 * Get instance
	 *
	 * @since 1.0.0
	 * @return instance object
	 */
	public static function get_instance() {
		return self::$_this;
	}

	/**
	 * Load admin scripts
	 *
	 * @since 1.0.0
	 * @return bool
	 */
	public function load_scripts() {
		$screen = get_current_screen();

		$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

		// load script only on settings products tab
		if ( $screen->base === 'woocommerce_page_wc-settings' && isset( $_GET['tab'] ) && $_GET['tab'] === 'products' ) {
			wp_enqueue_script( 'wc-store-catalog-pdf-download-admin-js', WC_STORE_CATALOG_PDF_DOWNLOAD_URL . 'assets/js/admin' . $suffix . '.js', array( 'jquery' ), WC_STORE_CATALOG_PDF_DOWNLOAD_VERSION, true );

			wp_enqueue_media();

			$localized_vars = array(
				'modalLogoTitle'  => __( 'Add Logo Image', 'woocommerce-store-catalog-pdf-download' ),
				'buttonLogoText'  => __( 'Upload Image', 'woocommerce-store-catalog-pdf-download' ),
				'modalPDFTitle'   => __( 'Add PDF', 'woocommerce-store-catalog-pdf-download' ),
				'buttonPDFText'   => __( 'Upload PDF', 'woocommerce-store-catalog-pdf-download' ),
				'removeImage'     => __( 'Are you sure you want to remove this image?', 'woocommerce-store-catalog-pdf-download' ),
				'removePDF'       => __( 'Are you sure you want to remove this PDF?', 'woocommerce-store-catalog-pdf-download' ),
				'previewLinkText' => __( 'Custom PDF Preview Link', 'woocommerce-store-catalog-pdf-download' ),
			);

			wp_localize_script( 'wc-store-catalog-pdf-download-admin-js', 'wc_store_catalog_pdf_download_admin_local', $localized_vars );

			wp_enqueue_style( 'wc-store-catalog-pdf-download-admin-css', WC_STORE_CATALOG_PDF_DOWNLOAD_URL . 'assets/css/admin-styles.css', array(), WC_STORE_CATALOG_PDF_DOWNLOAD_VERSION );
		}

		return true;
	}

	/**
	 * Add settings section to products tab
	 *
	 * @since 1.0.0
	 *
	 * @param array $sections existing sections
	 * @return array $sections modified sections
	 */
	public function add_settings_section( $sections ) {

		$sections['wc_store_catalog_pdf_download'] = __( 'Store Catalog PDF', 'woocommerce-store-catalog-pdf-download' );

		return $sections;
	}

	/**
	 * Add admin settings
	 *
	 * @since 1.0.0
	 *
	 * @param array  $settings existing settings
	 * @param string $current_section current section name
	 * @return array $settings
	 */
	public function add_settings( $settings, $current_section ) {
		if ( 'wc_store_catalog_pdf_download' === $current_section ) {
			return array(
				array(
					'title' => __( 'Store Catalog PDF', 'woocommerce-store-catalog-pdf-download' ),
					'id'    => 'woocommerce_store_catalog_pdf_download_settings',
					'type'  => 'woocommerce_store_catalog_pdf_download_settings',
				),
			);
		}

		return $settings;
	}

	/**
	 * Get admin settings
	 *
	 * @since 1.0.0
	 *
	 * @param array  $settings existing settings
	 * @param string $current_section current section name
	 */
	public function get_settings() {
		global $current_section;

		if ( 'wc_store_catalog_pdf_download' !== $current_section ) {
			return;
		}

		$help_icon = esc_url( WC()->plugin_url() ) . '/assets/images/help.png';

		$custom_pdf = get_option( 'wc_store_catalog_pdf_download_custom_pdf', '' );

		$logo = get_option( 'wc_store_catalog_pdf_download_logo', '' );

		$show_header = get_option( 'wc_store_catalog_pdf_download_show_header', 'no' );

		$header_text = get_option( 'wc_store_catalog_pdf_download_header_text', '' );

		$show_footer = get_option( 'wc_store_catalog_pdf_download_show_footer', 'no' );

		$footer_text = get_option( 'wc_store_catalog_pdf_download_footer_text', '' );

		$layout = get_option( 'wc_store_catalog_pdf_download_layout', 'list' );

		$link_label = get_option( 'wc_store_catalog_pdf_download_link_label', __( 'Download Catalog', 'woocommerce-store-catalog-pdf-download' ) );

		$hide_header_text = '';

		if ( 'no' === $show_header ) {
			$hide_header_text = 'display:none;';
		}

		$hide_footer_text = '';

		if ( 'no' === $show_footer ) {
			$hide_footer_text = 'display:none;';
		}

		// custom pdf
		$hide_remove_pdf_link = '';

		if ( ! empty( $custom_pdf ) ) {
			$custom_pdf_url = wp_get_attachment_url( $custom_pdf );

		} else {
			$custom_pdf_url       = '';
			$hide_remove_pdf_link = 'display:none;';
		}

		// logo image
		$hide_remove_image_link = '';
		$hide_preview_image     = '';

		$logo_image_url = wp_get_attachment_image_src( $logo, 'full' );

		if ( empty( $logo_image_url ) ) {
			$hide_remove_image_link = 'display:none;';
			$hide_preview_image     = ' hide';
		}
		?>
		<h3><?php esc_html_e( 'Store Catalog PDF', 'woocommerce-store-catalog-pdf-download' ); ?></h3>

		<h4><?php esc_html_e( 'Ready-made PDF', 'woocommerce-store-catalog-pdf-download' ); ?></h4>

		<table class="form-table">
			<tbody>
				<tr valign="top">
					<th class="titledesc" scope="row">
						<?php esc_html_e( 'Custom PDF', 'woocommerce-store-catalog-pdf-download' ); ?>
						<img class="help_tip" data-tip="<?php echo wc_sanitize_tooltip( __( 'Upload a ready made PDF of your store.', 'woocommerce-store-catalog-pdf-download' ) ); ?>" src="<?php echo esc_url( $help_icon ); ?>" height="16" width="16"/>
					</th>

					<td class="forminp">
						<a href="#" class="wc-store-catalog-pdf-download-upload-custom-pdf button"><?php esc_html_e( 'Upload Custom PDF', 'woocommerce-store-catalog-pdf-download' ); ?></a>
						<p class="description"><?php esc_html_e( 'Optional: This is to be used with the shortcode to display a ready made PDF', 'woocommerce-store-catalog-pdf-download' ); ?></p>

						<input type="hidden" name="wc_store_catalog_pdf_download_custom_pdf" value="<?php echo esc_attr( $custom_pdf ); ?>" id="wc_store_catalog_pdf_download_custom_pdf"/>

						<br/>

						<?php if ( $custom_pdf ) { ?>
							<a href="<?php echo esc_url( $custom_pdf_url ); ?>" target="_blank" class="custom-pdf-preview"><?php esc_html_e( 'Custom PDF Preview Link', 'woocommerce-store-catalog-pdf-download' ); ?></a>

						<?php } else { ?>
							<a href="#" target="_blank" class="custom-pdf-preview"></a>

						<?php } ?>

						<a href="#" class="remove-pdf dashicons dashicons-no" style="<?php echo esc_attr( $hide_remove_pdf_link ); ?>" title="<?php esc_attr_e( 'Click to remove PDF', 'woocommerce-store-catalog-pdf-download' ); ?>"></a>
					</td>
				</tr>
			</tbody>
		</table>

		<h4><?php esc_html_e( 'PDF Generator', 'woocommerce-store-catalog-pdf-download' ); ?></h4>

		<table class="form-table">
			<tbody>
				<tr valign="top">
					<th class="titledesc" scope="row">
						<?php esc_html_e( 'Company Logo', 'woocommerce-store-catalog-pdf-download' ); ?>
						<img class="help_tip" data-tip="<?php echo wc_sanitize_tooltip( __( 'Upload a logo to be shown at the top of the PDF.', 'woocommerce-store-catalog-pdf-download' ) ); ?>" src="<?php echo esc_url( $help_icon ); ?>" height="16" width="16"/>
					</th>

					<td class="forminp">
						<a href="#" class="wc-store-catalog-pdf-download-upload-logo button"><?php esc_html_e( 'Upload Logo', 'woocommerce-store-catalog-pdf-download' ); ?></a>

						<input type="hidden" name="wc_store_catalog_pdf_download_logo" value="<?php echo esc_attr( $logo ); ?>" id="wc_store_catalog_pdf_download_logo"/>

						<br/>

						<?php if ( is_array( $logo_image_url ) && ! empty( $logo_image_url ) ) { ?>
							<img src="<?php echo esc_url( $logo_image_url[0] ); ?>" class="logo-preview-image"/>

						<?php } else { ?>
							<img src="" class="logo-preview-image<?php echo esc_attr( $hide_preview_image ); ?>"/>

						<?php } ?>

						<a href="#" class="remove-image dashicons dashicons-no" style="<?php echo esc_attr( $hide_remove_image_link ); ?>" title="<?php esc_attr_e( 'Click to remove image', 'woocommerce-store-catalog-pdf-download' ); ?>"></a>
					</td>
				</tr>

				<tr valign="top">
					<th class="titledesc" scope="row"><?php esc_html_e( 'PDF Header', 'woocommerce-store-catalog-pdf-download' ); ?></th>
					<td class="forminp">
						<label for="show-header"><input type="checkbox" id="show-header" name="wc_store_catalog_pdf_download_show_header" <?php checked( $show_header, 'yes' ); ?> /> <?php esc_html_e( 'Show Header', 'woocommerce-store-catalog-pdf-download' ); ?></label>
						<p class="description"><?php esc_html_e( 'Enabling this option will show a header information at the top of the PDF such as any intro verbiage.', 'woocommerce-store-catalog-pdf-download' ); ?></p>
					</td>
				</tr>

				<tr valign="top" class="header-text-row" style="<?php echo esc_attr( $hide_header_text ); ?>">
					<th class="titledesc" scope="row"><?php esc_html_e( 'PDF Header Text', 'woocommerce-store-catalog-pdf-download' ); ?></th>
					<td class="forminp">
						<p><textarea name="wc_store_catalog_pdf_download_header_text" style="width:100%;height:200px;"><?php echo wp_kses_post( $header_text ); ?></textarea></p>
					</td>
				</tr>

				<tr valign="top">
					<th class="titledesc" scope="row"><?php esc_html_e( 'PDF Footer', 'woocommerce-store-catalog-pdf-download' ); ?></th>
					<td class="forminp">
						<label for="show-footer"><input type="checkbox" id="show-footer" name="wc_store_catalog_pdf_download_show_footer" <?php checked( $show_footer, 'yes' ); ?> /> <?php esc_html_e( 'Show Footer', 'woocommerce-store-catalog-pdf-download' ); ?></label>
						<p class="description"><?php esc_html_e( 'Enabling this option will show a footer information at the bottom of the PDF such as foot notes or any outro verbiage.', 'woocommerce-store-catalog-pdf-download' ); ?></p>
					</td>
				</tr>

				<tr valign="top" class="footer-text-row" style="<?php echo esc_attr( $hide_footer_text ); ?>">
					<th class="titledesc" scope="row"><?php esc_html_e( 'PDF Footer Text', 'woocommerce-store-catalog-pdf-download' ); ?></th>
					<td class="forminp">
						<p><textarea name="wc_store_catalog_pdf_download_footer_text" style="width:100%;height:200px;"><?php echo wp_kses_post( $footer_text ); ?></textarea></p>
					</td>
				</tr>

				<tr valign="top">
					<th class="titledesc"
						scope="row"><?php esc_html_e( 'PDF Layout Format', 'woocommerce-store-catalog-pdf-download' ); ?>
						<img class="help_tip" data-tip="<?php echo wc_sanitize_tooltip( __( 'Set how you want the PDF layout format to display for category products.', 'woocommerce-store-catalog-pdf-download' ) ); ?>" src="<?php echo esc_url( $help_icon ); ?>" height="16" width="16"/>
					</th>
					<td class="forminp">
						<select name="wc_store_catalog_pdf_download_layout" class="wc-enhanced-select">
							<option value="list" <?php selected( $layout, 'list' ); ?>><?php esc_html_e( 'List Format', 'woocommerce-store-catalog-pdf-download' ); ?></option>
							<option value="grid" <?php selected( $layout, 'grid' ); ?>><?php esc_html_e( 'Grid Format', 'woocommerce-store-catalog-pdf-download' ); ?></option>
						</select>
					</td>
				</tr>

				<tr valign="top">
					<th class="titledesc" scope="row"><?php esc_html_e( 'Download Link Label', 'woocommerce-store-catalog-pdf-download' ); ?>
						<img class="help_tip" data-tip="<?php echo wc_sanitize_tooltip( __( 'Set the text you want to display next to the download icon.', 'woocommerce-store-catalog-pdf-download' ) ); ?>" src="<?php echo esc_url( $help_icon ); ?>" height="16" width="16"/>
					</th>
					<td class="forminp">
						<input type="text" name="wc_store_catalog_pdf_download_link_label" value="<?php echo esc_attr( $link_label ); ?>"/>
					</td>
				</tr>
			</tbody>
		</table>
		<?php
	}

	/**
	 * Save admin settings
	 *
	 * @since 1.0.0
	 *
	 * @return bool true
	 */
	public function save_settings() {
		global $current_section;

		if ( 'wc_store_catalog_pdf_download' !== $current_section ) {
			return true;
		}

		// phpcs:disable WordPress.Security.NonceVerification
		update_option( 'wc_store_catalog_pdf_download_show_header', wc_bool_to_string( isset( $_POST['wc_store_catalog_pdf_download_show_header'] ) ) );
		update_option( 'wc_store_catalog_pdf_download_show_footer', wc_bool_to_string( isset( $_POST['wc_store_catalog_pdf_download_show_footer'] ) ) );

		$setting_keys = array(
			'wc_store_catalog_pdf_download_custom_pdf',
			'wc_store_catalog_pdf_download_logo',
			'wc_store_catalog_pdf_download_header_text',
			'wc_store_catalog_pdf_download_footer_text',
			'wc_store_catalog_pdf_download_layout',
			'wc_store_catalog_pdf_download_link_label',
		);

		foreach ( $setting_keys as $key ) {
			if ( ! isset( $_POST[ $key ] ) ) {
				continue;
			}

			switch ( $key ) {
				case 'wc_store_catalog_pdf_download_header_text':
				case 'wc_store_catalog_pdf_download_footer_text':
					$value = wp_kses_post( wp_unslash( $_POST[ $key ] ) );
					break;
				default:
					$value = sanitize_text_field( wp_unslash( $_POST[ $key ] ) );
					break;
			}

			update_option( $key, $value );
		}
		// phpcs:enable WordPress.Security.NonceVerification

		return true;
	}

	/**
	 * Renders the debug fields.
	 *
	 * @return bool
	 * @deprecated 2.2.0
	 *
	 * @since 1.0.0
	 */
	public function render_debug_fields() {
		wc_deprecated_function( __FUNCTION__, '2.2.0', '\KoiLab\WC_Store_Catalog_PDF_Download\Admin\System_Status_Report::output_content()' );

		return true;
	}
}

new WC_Store_Catalog_PDF_Download_Admin();
